%**************************************************************************
% readSPM32_DC_IV_ave_dif_plot.m
% 20 Feb 2007
% Purpose: to process RHK SPM32 IV curves created with
%  Point Spectroscopy->DC-IV
%  averages IV curves
%  differentiates this average
% Returns: array A, raw IV data from sm2 file
% Input Parameters: order, the order of polynomial to fit to IV data
%
% RHK is a structure with data from SPM32's SM2 file format header
% taken from RHK SPM 32 version 2004 User's Guide
% RHK structure field names correspond to those of SM2 file format
%
% These must be true in SPM32 header,
%   RHK.sub_type = 7 (see RHK manual header format under line_type,
%                       where 7 = I-V spectrum )
% For I-V spectra, the following variables have a different meaning,
%   RHK.x_size = number of points per spectra line
%   RHK.y_size = number of spectra in page
%   RHK.
%   X data is volts
%   Z data is current
%   Y data is curve number
%
%   X data is calculated with RHK.xoffset as first voltage value
%   and voltage step RHK.xscale
%   Z data is read from file into integer array A[RHK.x_size,RHK.y_size],
%   scaled to current with RHK.zscale & RHK.zoffset and stored in array B
%
%**************************************************************************

%**************************************************************************
% primary function
%**************************************************************************
function A = readSPM32_DC_IV_ave_dif_plot(window, filename, pathname)
%    cd('C:\Documents and Settings\msm\My Documents\L436\');
%    [filename, pathname] = uigetfile('*.sm2', 'Pick an SM2-file');
%    if isequal(filename,0)
%        disp('User selected Cancel')
%    else
        disp(['User selected', fullfile(pathname, filename)])
        % extract filename root for output files
        filenameLen = length(filename);
        filenameRoot = filename(1:filenameLen-4); % remove '.sm2'
        % open data file
        filenameR = sprintf('%s%s', pathname, filename);
        fid = fopen(filenameR,'r');

        % initialize file positioning variables
        file_pointer = 0;
        fileStatus = 0;
        RHK.sub_type = 0; 
        RHK.size = 0;
        Adim = 0;

        % search for I-V spectrum data
        fileStatus = fseek(fid,file_pointer,'bof');
        while ~fileStatus
            % increment file_pointer to beginning of page header
            % will increment by 0 when RHK.size = 0 
            file_pointer = file_pointer + RHK.size
%            disp(file_pointer);
            % move to start of page header
            file_pointer = file_pointer + 1
            fileStatus = fseek(fid,file_pointer,'bof')
            if ~fileStatus
                file_pointer = file_pointer - 1
                fseek(fid,file_pointer,'bof');
                RHK = read_header(fid);
                % increment file_pointer to end of page header    
                file_pointer = file_pointer + RHK.data_offset
    %            disp(file_pointer);
                % read data block
                fileStatus = fseek(fid,file_pointer,'bof')
                Adim = Adim + 1
                A(:,:,Adim) = fread(fid,[RHK.x_size,RHK.y_size],'bit16');
            end
        end
        fclose(fid);

        if RHK.sub_type == 7
            % scale data 
            x = generate_Voltage_Vec(RHK);
            B = A * RHK.zscale + RHK.zoffset; % scale z to current data
            % fit polinomial after removing saturated data
            [firstNonSat, lastNonSat] = find_saturated_data(A,RHK);
%            disp(sprintf('polynomial order %d',order));
%            p = fit_Polynomial(B,x,order,firstNonSat,lastNonSat,RHK);
%            y = polyval(p,x);
 %           display_poly(p,order,'p');
 %           dp = derivative(p,order);
 %           s=polyval(dp,x);
 %           display_poly(dp,order-1,'dp');
            yAveraged = average_data(B,RHK);
            ySmoothed = smooth_data(yAveraged,window, RHK);
            ySmoothed_dIdV = numDiff(ySmoothed,window,x, RHK);
            % write data to files
            write_ySmoothed(pathname,filenameRoot,ySmoothed,x,window,RHK);
            write_ySmoothed_dI_dV(pathname,filenameRoot,ySmoothed_dIdV,x,window,RHK);
            
 %           write_IV_file(pathname,filenameRoot,x,y,RHK);
 %           write_dIdV_file(pathname,filenameRoot,x,s,RHK);
 %           write_ASCII_file(pathname,filenameRoot,B,x,y,s,RHK);
 
        end % RHK.sub_type == 7   
  
%    end %NOT isequal(filename,0)
%**************************************************************************
% END primary function
%**************************************************************************

%**************************************************************************
% function write_ySmoothed_dI_dV
%**************************************************************************
function write_ySmoothed_dI_dV(pathname,filenameRoot,ySmoothed_dIdV,x,window,RHK)
    filenameW = sprintf('%s%s.slp', pathname, filenameRoot);
    fid = fopen(filenameW,'w');
    for m = window+2: RHK.x_size-(window+1) %RHK.x_size is points per spectra
        fprintf(fid,'%8.4f \t%12.8e \n',x(m),ySmoothed_dIdV(m));
    end
    fclose(fid);
%**************************************************************************
% END function write_ySmoothed_dI_dV
%**************************************************************************

    
%**************************************************************************
% function numDiff
%**************************************************************************
function slope = numDiff(ySmoothed,window,x, RHK)
    slope = zeros(RHK.x_size-(window+1));
    for m = window+2: RHK.x_size-(window+1) %RHK.x_size is points per spectra
         slope(m) =  (ySmoothed(m) - ySmoothed(m-1))/(x(m) - x(m-1));
    end
    
    plot(x(window+2:RHK.x_size-(window+1)),slope(window+2:RHK.x_size-(window+1)),'-'), grid on
%**************************************************************************
% END function numDiff
%**************************************************************************

%**************************************************************************
% function write_ySmoothed
%**************************************************************************
function write_ySmoothed(pathname,filenameRoot,ySmoothed,x,window,RHK)
    filenameW = sprintf('%s%s.ysm', pathname, filenameRoot);
    fid = fopen(filenameW,'w');
    for m = window+1: RHK.x_size-(window+1) %RHK.x_size is points per spectra
        fprintf(fid,'%8.4f \t%12.8e \n',x(m),ySmoothed(m));
    end
    fclose(fid);
%**************************************************************************
% END function write_ySmoothed
%**************************************************************************

%**************************************************************************
% function average_data
%**************************************************************************
function yAveraged = average_data(B, RHK)
    yAveraged = zeros(RHK.x_size);
    for m = 1:RHK.x_size %RHK.x_size is points per spectra
        yAveraged(m) = 0.0;
        for n = 1:RHK.y_size %RHK.y_size is number of spectra
            yAveraged(m) = yAveraged(m) + B(m,n);
        end
        yAveraged(m) = yAveraged(m)/RHK.y_size;
    end
%**************************************************************************
% END function average_data
%**************************************************************************

%**************************************************************************
% function smooth_data
%**************************************************************************
function ySmoothed = smooth_data(yAveraged, window, RHK)
    ySmoothed = zeros(RHK.x_size-(window+1));
    for m = window+1: RHK.x_size-(window+1) %RHK.x_size is points per spectra
         ySmoothed(m) = 0.0;
        for n = m-window : m + window
             ySmoothed(m) =  ySmoothed(m)+ yAveraged(n);
        end
         ySmoothed(m) =  ySmoothed(m) / (2*window+1);
    end
%**************************************************************************
% END function smooth_data
%**************************************************************************


%**************************************************************************
% function display_poly
%**************************************************************************
function display_poly(p,order,name)
    textPoly = sprintf('%s = ',name);
    for n = order:-1:0
        textPoly = strcat(textPoly, sprintf(' %+8.6e*A1%c%d ',p(order-n+1),'^',n));
    end
    disp(textPoly);
%**************************************************************************
% END function display_poly
%**************************************************************************

%**************************************************************************
% function derivative
%**************************************************************************
function dp = derivative(p,order)
    dp = zeros(1,order);
    for n = 1:order
        dp(n) = p(n)*(order - n + 1);
    end
%**************************************************************************
% END function derivative
%**************************************************************************

%**************************************************************************
% function fit_Polynomial
%**************************************************************************
function p = fit_Polynomial(B,x,order,firstNonSat,lastNonSat,RHK)
    % average across remaining data
    y = zeros(lastNonSat-firstNonSat+1);
    for m = 1:(lastNonSat-firstNonSat+1)
        y(m) = 0.0;
        for n = 1:RHK.y_size 
            y(m)= y(m) + B(m+firstNonSat-1,n);
        end
        y(m) = y(m)/RHK.y_size;
    end
    X = x(firstNonSat:lastNonSat);
    Y = y(firstNonSat:lastNonSat);
    p=polyfit(X,Y,order);
%**************************************************************************
% END function fit_Polynomial
%**************************************************************************

%**************************************************************************
% function find_saturated_data
%**************************************************************************
function [firstNonSat, lastNonSat] = find_saturated_data(A,RHK)
    firstNonSat = -1;
    lastNonSat = -1;
    for m = 1:RHK.x_size
        foundSat = false;
        for n = 1:RHK.y_size
            if (A(m,n) == 2047)
                foundSat = true;
                break;
            end  
        end
        if foundSat == false
            firstNonSat = m;
            break
        end
    end
    for m = RHK.x_size:-1:1
        foundSat = false;
        for n = 1:RHK.y_size
            if (A(m,n) == -2048)
                foundSat = true;
                break;
            end  
        end
        if foundSat == false
            lastNonSat = m;
            break
        end
    end
%**************************************************************************
% END function find_saturated_data
%**************************************************************************

%**************************************************************************
% function generate_Voltage_Vec
%**************************************************************************
function x = generate_Voltage_Vec(RHK)
    x = zeros(RHK.x_size);
    x(1)= RHK.xoffset;
    for m = 2:RHK.x_size %RHK.x_size is points per spectra
        x(m) = x(m-1)+RHK.xscale;
    end
%**************************************************************************
% END function generate_Voltage_Vec
%**************************************************************************

%**************************************************************************
% function write_ASCII_file
%**************************************************************************
function B = write_ASCII_file(pathname,filenameRoot,B,x,y,s,RHK)
    filenameW = sprintf('%s%s.dat', pathname, filenameRoot);
    fid = fopen(filenameW,'w');
    for m = 1:RHK.x_size %RHK.x_size is points per spectra
        fprintf(fid,'%8.4f ',x(m));
        for n = 1:RHK.y_size %RHK.y_size is number of spectra
            fprintf(fid,'\t%12.8e ',B(m,n));
        end
        fprintf(fid,'\n');
    end
    status = fclose(fid);
%**************************************************************************
% END function write_ASCII_file
%**************************************************************************

%**************************************************************************
% function write_IV_file
%**************************************************************************
function write_IV_file(pathname,filenameRoot,x,y,RHK)
    filenameW = sprintf('%s%s.iv', pathname, filenameRoot);
    fid = fopen(filenameW,'w');
    for m = 1:RHK.x_size %RHK.x_size is points per spectra
        fprintf(fid,'%8.4f\t%12.8e\n',x(m),y(m));
    end
    status = fclose(fid);
%**************************************************************************
% END function write_IV_file
%**************************************************************************

%**************************************************************************
% function write_dIdV_file
%**************************************************************************
function write_dIdV_file(pathname,filenameRoot,x,s,RHK)
    filenameW = sprintf('%s%s.div', pathname, filenameRoot);
    fid = fopen(filenameW,'w');
    for m = 1:RHK.x_size %RHK.x_size is points per spectra
        fprintf(fid,'%8.4f\t%12.8e\n',x(m),s(m));
    end
    status = fclose(fid);
%**************************************************************************
% END function write_dIdV_file
%**************************************************************************

%**************************************************************************
% function read_header
%**************************************************************************
 function RHK = read_header(fid)
    RHK.id = fscanf(fid, '%11c', 1 );
    disp(RHK.id);
    RHK.date = fscanf(fid, '%9c', 1 );
    disp(RHK.date);
    RHK.time = fscanf(fid, '%9c', 1 );
%    disp(RHK.time);
    RHK.type = fscanf(fid, '%d', 1 );
%    disp(RHK.type);
    RHK.data_type = fscanf(fid, '%d', 1 );
%    disp(RHK.data_type);
    RHK.sub_type = fscanf(fid, '%d', 1 );
%    disp(RHK.sub_type);
    RHK.x_size = fscanf(fid, '%d', 1 );
%    disp(RHK.x_size);
    RHK.y_size = fscanf(fid, '%d', 1 );
%    disp(RHK.y_size);
    RHK.size = fscanf(fid, '%d', 1 );
%    disp(RHK.size);
    RHK.page_type = fscanf(fid, '%d', 1 );
%    disp(RHK.page_type);
    %
    u_text = fscanf(fid, '%s', 1 );
%    disp(u_text);
    RHK.xscale = fscanf(fid, '%e', 1 );
%    disp(RHK.xscale);
    RHK.xoffset = fscanf(fid, '%e', 1 );
%    disp(RHK.xoffset);
    RHK.xunits = fscanf(fid, '%9c', 1 );
%    disp(RHK.xunits);
    %
    u_text = fscanf(fid, '%s', 1 );
%    disp(u_text);
    RHK.yscale = fscanf(fid, '%e', 1 );
%    disp(RHK.yscale);
    RHK.yoffset = fscanf(fid, '%e', 1 );
%    disp(RHK.yoffset);
    RHK.yunits = fscanf(fid, '%9c', 1 );
%    disp(RHK.yunits);
    %
    u_text = fscanf(fid, '%s', 1 );
%    disp(u_text);
    RHK.zscale = fscanf(fid, '%e', 1 );
%    disp(RHK.zscale);
    RHK.zoffset = fscanf(fid, '%e', 1 );
%    disp(RHK.zoffset);
    RHK.zunits = fscanf(fid, '%10s', 1 );
%    disp(RHK.zunits);
    %
    u_text = fscanf(fid, '%s', 1 );
%    disp(u_text);
    RHK.xyscale = fscanf(fid, '%e', 1 );
%    disp(RHK.xyscale);
    RHK.xy = fscanf(fid, '%s', 1 );
%    disp(RHK.xy);
    RHK.angle = fscanf(fid, '%e', 1 );
%    disp(RHK.angle);
    %
    u_text = fscanf(fid, '%s', 1 );
%    disp(u_text);
    RHK.current = fscanf(fid, '%e', 1 );
%    disp(RHK.current);
    RHK.bias = fscanf(fid, '%e', 1 );
%    disp(RHK.bias);
    %
    u_text = fscanf(fid, '%s', 1 );
%    disp(u_text);
    RHK.scan = fscanf(fid, '%d', 1 );
%    disp(RHK.scan);
    RHK.period = fscanf(fid, '%e', 1 );
%    disp(RHK.period);
    %
    u_text = fscanf(fid, '%s', 1 );
%    disp(u_text);
    RHK.file_id = fscanf(fid, '%d', 1 );
%    disp(RHK.file_id);
    RHK.data_offset = fscanf(fid, '%d', 1 );
%    disp(RHK.data_offset);
    %
    RHK.label = fscanf(fid, '%20s', 1 );
%    disp(RHK.label);
    RHK.text = fscanf(fid, '%160s', 1 );
%    disp(RHK.text);
%**************************************************************************
% END function read_header
%**************************************************************************

    %**************************************************************************
    %**************************************************************************
    % average data
    %**************************************************************************
    %**************************************************************************
    
        
        

     %{   
        x = A(:,1);
        windowSize = 15;
        %y = filter(ones(1,windowSize)/windowSize,1,x);
        halfWindowSize = round(windowSize/2)-1;
        nBeg = 1+halfWindowSize;
        nEnd = length(x)-halfWindowSize;
        %nBeg = windowSize;
        %nEnd = length(x)-windowSize+1;
        for n = nBeg:nEnd
            y(n)=0.0;
        %    disp(n);
            for m = (n-halfWindowSize):(n+halfWindowSize)
        %        disp(m);
                y(n)= y(n) + x(m);
            end
            y(n)= y(n)/windowSize;
        end
        t = nBeg:nEnd;
        plot(t,x(nBeg:nEnd),'-.',t,y(nBeg:nEnd),'-'), grid on
    %} 
    %**************************************************************************
    %**************************************************************************

